#include "toolbar.h"

ToolBar::ToolBar(bool vertical, bool fullWidth, QWidget *parent, bool hideContainer) : QWidget(parent)
{
	setWindowIcon( QIcon(":librework.png") );
	setAttribute( Qt::WA_TranslucentBackground );
	//setStyleSheet("QWidget { background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1, stop: 0 #55aaaa, stop: 1 #66bbbb); }");
	//setStyleSheet("QWidget { border-radius: 10px; border-style: inset; background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1, stop: 0 #aaaaaa, stop: 1 #cccccc); }");
	setStyleSheet("ToolBar { background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 1, stop: 0 #aaaaaa, stop: 1 #cccccc); }");

	timer = new QTimer(this);
	visible = false;
	this->fullWidth = fullWidth;
	this->vertical = vertical;
	this->hideContainer = hideContainer;

	if ( vertical )
	{
		layout = new QVBoxLayout();
	} else {
		layout = new QHBoxLayout();
	}
	layout->setContentsMargins(0, 1, 0, 0);
	startSpacer = new QSpacerItem(32768, BUTTON_SIZE);
	endSpacer = new QSpacerItem(32768, BUTTON_SIZE);
	layout->addSpacerItem( startSpacer );
	layout->addSpacerItem( endSpacer );

	_menuButton = new QPushButton(QIcon(":menu.png"),"");
	addButton( _menuButton );

	setMenu();

	connect( _menuButton, SIGNAL(clicked()), this, SLOT(menuButtonClicked()));

	this->setLayout( layout );
}

ToolBar::~ToolBar()
{
	delete timer;
	delete layout;
}

void ToolBar::setMenu()
{
	menu = new QMenu;
	menu->addAction( QIcon(":librework-256.png"), tr("About Librework Client"), this, SLOT( showAbout() ) );

	QProcess monitorCount;
	monitorCount.start( MULTIMONSCRIPT, QStringList("screencount") );
	monitorCount.waitForStarted();
	monitorCount.waitForFinished();
	bool ok;
	QByteArray output = monitorCount.readAllStandardOutput().simplified();
	int count = output.toInt( &ok );
	if ( ok && count >= 2 )
	{
		menu->addAction( QIcon(":librework-256.png"), tr("Set monitors duplicated"), this, SLOT( duplicateScreen() ) );
		menu->addAction( QIcon(":librework-256.png"), tr("Set monitors extended"), this, SLOT( extendScreen() ) );
	}
	QAction * showAdvanced = new QAction( tr("Show advanced options") );
	showAdvanced->setCheckable( true );
	menu->addAction( showAdvanced );
	QAction * disconnectAction = menu->addAction( QIcon(":suspend.png"), tr("Suspend session"), this, SIGNAL( disconnectSession() ) );
	disconnectAction->setVisible( false );
	if ( ! hideContainer ) // when called from Librework Desktop
	{
		 menu->addAction( QIcon(":minimize.png"), tr("Minimize window"), this, SIGNAL( minimizeWindow() ) );
	}
	connect( showAdvanced, SIGNAL( toggled(bool) ), disconnectAction, SLOT( setVisible(bool) ) );
	connect( showAdvanced, SIGNAL( toggled(bool) ), this, SLOT( menuButtonClicked() ) );
}

void ToolBar::showAbout()
{
	QMessageBox::about( this, tr("About Librework Client"), tr("This is Librework Client v%1\n\n(2018) All rights reserved").arg( VERSION ) );
}

void ToolBar::duplicateScreen()
{
	QProcess multimon;
	multimon.start( MULTIMONSCRIPT, QStringList("duplicate") );
	multimon.waitForStarted();
	multimon.waitForFinished();
	emit refreshConnection();
}

void ToolBar::extendScreen()
{
	QProcess multimon;
	multimon.start( MULTIMONSCRIPT, QStringList("extend") );
	multimon.waitForStarted();
	multimon.waitForFinished();
	emit refreshConnection();
}

void ToolBar::menuButtonClicked()
{
	QRect screenrect = qApp->primaryScreen()->geometry();
	menu->show();
	menu->move( screenrect.right() - menu->width(), screenrect.top() + this->height() );
}

void ToolBar::setRoundedCorners(int radius)
{
    QRegion region(0, 0, width(), height(), QRegion::Rectangle);

    // bottom right
    QRegion round = QRegion(width()-2*radius, height()-2*radius, 2*radius, 2*radius, QRegion::Ellipse);
    QRegion corner = QRegion(width()-radius, height()-radius, radius, radius, QRegion::Rectangle);
    region = region.subtracted(corner.subtracted(round));

    // bottom left
    round = QRegion(0, height()-2*radius, 2*radius, 2*radius, QRegion::Ellipse);
    corner = QRegion(0, height()-radius, radius, radius, QRegion::Rectangle);
    region = region.subtracted(corner.subtracted(round));

    setMask(region);
}

void ToolBar::addButton(QPushButton *button)
{
    layout->removeItem( endSpacer );
    button->setFixedHeight( BUTTON_SIZE );
    button->setFixedWidth( BUTTON_SIZE );
    button->setIconSize(QSize(BUTTON_SIZE - BAR_MARGIN, BUTTON_SIZE - BAR_MARGIN));
    button->setFlat( true );
    if ( button == _menuButton )
    {
        layout->addSpacerItem( endSpacer );
        layout->addWidget( button );
    } else {
        layout->removeWidget( _menuButton );
        layout->addWidget( button );
        layout->addSpacerItem( endSpacer );
        layout->addWidget( _menuButton );
    }
}

void ToolBar::enterEvent ( QEvent * )
{
	if ( ! fullWidth )
	{
		setMouseTracking( true );
	}
}

void ToolBar::leaveEvent ( QEvent * )
{
	if ( visible && ! fullWidth )
	{
		connect( timer, SIGNAL(timeout()), this, SLOT(timeFinished()));
		timer->start(TIME_LIMIT);
	}
}

void ToolBar::mouseMoveEvent (QMouseEvent * mouseEvent)
{
    if (timer->isActive())
    {
        timer->stop();
        disconnect( timer, SIGNAL(timeout()), this, SLOT(timeFinished()));
    }

    if (!visible && mouseEvent->x() >= WIDTH_LIMIT && mouseEvent->x() <= width() - WIDTH_LIMIT)
    {
        visible = true;
        setMouseTracking(false);
        for (int i = 1; i <= BAR_HEIGHT; i = i + BAR_HEIGHT / ANIMATION_STEPS)
        {
            setFixedHeight(i);
#if QT_VERSION < 0x050000
#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
            usleep(ANIMATION_STEP_TIME * 1000);
#endif
#else
            QThread::msleep(ANIMATION_STEP_TIME);
#endif
            QApplication::processEvents();
        }
        setFixedHeight(BAR_HEIGHT);
        setRoundedCorners(10);
        setMouseTracking(true);
        emit entered();
    }
}

void ToolBar::paintEvent ( QPaintEvent * )
{
    QStyleOption opt;
    opt.initFrom(this);
    QPainter p(this);
    style()->drawPrimitive(QStyle::PE_Widget, &opt, &p, this);
}

void ToolBar::timeFinished()
{
    emit left();
    setMouseTracking( false );
    visible = false;
    timer->stop();
    disconnect( timer, SIGNAL(timeout()), this, SLOT(timeFinished()));
    for (int i = BAR_HEIGHT; i >= 1; i = i - BAR_HEIGHT / ANIMATION_STEPS)
    {
        setFixedHeight(i);
#if QT_VERSION < 0x050000
#if defined(Q_OS_LINUX) || defined(Q_OS_FREEBSD)
            usleep(ANIMATION_STEP_TIME * 1000);
#endif
#else
        QThread::msleep(ANIMATION_STEP_TIME);
#endif
        QApplication::processEvents();
    }
    setFixedHeight(1);
}
