#include "clientsettings.h"

ClientSettings::ClientSettings()
{
    userSettings = new QSettings(QSettings::IniFormat, QSettings::UserScope, "librework", "librework-client");
	systemSettings = new QSettings(CLIENT_CONF_DIR "/librework-client.ini", QSettings::IniFormat);

    fixClientSettings();
}

ClientSettings::~ClientSettings()
{
    delete userSettings;
    delete systemSettings;
}

void ClientSettings::sync()
{
    userSettings->sync();
    systemSettings->sync();
}

void ClientSettings::fixClientSettings()
{
    QStringList systemList = systemSettings->childGroups();
    QStringList userList = userSettings->childGroups();

    for (int i = 0; i < systemList.count(); i++)
    {
        if (userList.contains(systemList[i]))
        {
            rename(systemList[i], QObject::tr("user") + "_" + systemList[i]);
        }
    }
}

void ClientSettings::rename(QString oldName, QString newName)
{
    userSettings->beginGroup(oldName);
    QStringList keys = userSettings->childKeys();
    userSettings->endGroup();
    for (int j = 0; j < keys.count(); j++)
    {
        userSettings->setValue( newName + "/" + keys[j], userSettings->value(oldName + "/" + keys[j]));
    }
    userSettings->remove( oldName );
}

int ClientSettings::addServer(QString name)
{
    if (serverList().contains(name))
    {
        return EXISTING_NAME;
    }
    if (name.contains("/"))
    {
        return BAD_NAME;
    }
    setType(name, "RDP NLA");
    return OK_NAME;
}

QStringList ClientSettings::serverTypes()
{
    QStringList res;
    res += "RDP NLA";
    res += "RDP TLS";
    res += "RDP";

    return res;
}

QStringList ClientSettings::screenSizes()
{
    QStringList res;
    res += QObject::tr("Full Screen");
    res += QObject::tr("Maximized window");
    res += "1920x1080";
    res += "1600x1050";
    res += "1600x900";
    res += "1280x1024";
    res += "1200x800";
    res += "1024x768";
    res += "1024x600";
    res += "800x600";
    res += "800x480";
    res += "640x480";

    return res;
}

QStringList ClientSettings::colorDepths()
{
	QStringList res;
	res += "32";
	res += "24";
	res += "16";
	res += "8";

	return res;
}

QStringList ClientSettings::rfxModes()
{
	QStringList res;
	res += "Video";
	res += "Image";
	res += "Disable";

	return res;
}

bool ClientSettings::removeServer(QString name)
{
    if (userSettings->childGroups().contains(name))
    {
        userSettings->remove(name);
        return true;
    } else {
        return false;
    }
}

QStringList ClientSettings::defaultServersList()
{
    QStringList res;
    //res += "Librework Free";
    //res += "Librework Premium";

    return res;
}

QStringList ClientSettings::serverList()
{
    QStringList res;
    QStringList serverList = systemSettings->childGroups() + userSettings->childGroups();

    for (int i = 0; i < serverList.count(); i++)
    {
        if ( ! serverList[i].startsWith(APP_PREFIX, Qt::CaseInsensitive))
        {
            res += serverList[i];
        }
    }

    return res;
}

QStringList ClientSettings::userServerList()
{
    return userSettings->childGroups();
}

QVariant ClientSettings::getValue(QString name, QString key, QVariant defaultValue)
{
    if (userSettings->childGroups().contains(name))
    {
        return userSettings->value(name + "/" + key, defaultValue);
    } else {
        return systemSettings->value(name + "/" + key, defaultValue);
    }
}

QIcon ClientSettings::getIcon(QString name)
{
    QString connType = getType(name);
    if ( connType == "Librework" )
    {
        return QIcon(":/librework-logo-144.png");
    } else {
        return QIcon(":/rdp.png");
    }
}

QString ClientSettings::getLastLogin()
{
    return userSettings->value("LastLogin").toString();
}

QString ClientSettings::getLastServer()
{
    return userSettings->value("LastServer").toString();
}

QString ClientSettings::getLogin(QString name)
{
    return getValue(name, "Login").toString();
}

QString ClientSettings::getServer(QString name)
{
    if ( name == FREE_NAME)
        return QString(FREE_SERVER);

    if ( name == PREMIUM_NAME)
        return QString(PREMIUM_SERVER);

    if ( serverList().contains( name ) )
    {
        return getValue(name, "Server").toString();
    } else {
        return name.section( ':', 0, 0 );
    }
}

QString ClientSettings::getType(QString name)
{
    if ( name == FREE_NAME || name == PREMIUM_NAME)
        return QString("Librework");

    return getValue(name, "Type", "").toString();
}

bool ClientSettings::canCheckAuth(QString name)
{
	return getValue(name, "EnableAuthChecking", QVariant(false)).toBool();
}

QString ClientSettings::getDomain(QString name)
{
    return getValue(name, "Domain").toString();
}

long ClientSettings::getPort(QString name)
{
    if ( serverList().contains( name ) )
    {
        return getValue(name, "Port", RDP_DEFAULT_PORT).toLongLong();
    } else {
        QString portString = name.section( ':', 1, 1 );
        bool ok;
        long port = portString.toLong( &ok );
        if ( ok )
        {
            return port;
        } else {
            return RDP_DEFAULT_PORT;
        }
    }
}

QString ClientSettings::getScreenSize(QString name)
{
	QVariant screenSize = getValue(name, "ScreenSize", QVariant( "fullscreen" ));
	if ( screenSize.typeName() == QString("QSize") )
	{
		QSize qSize = getValue(name, "ScreenSize", QSize(0,0)).toSize();
		if ( qSize == QSize(0,0) )
		{
			return QString("fullscreen");
		} else {
			return QString::number( qSize.width() ) + "x" + QString::number( qSize.height() );
		}
	} else {
		return screenSize.toString();
	}
}

QString ClientSettings::getOrientation(QString name)
{
	QVariant orientation = getValue(name, "Orientation", QVariant( "horizontal" ));
	return orientation.toString();
}

int ClientSettings::getColorDepth(QString name)
{
    return getValue(name, "ColorDepth", 24).toInt();
}

QString ClientSettings::getDriveRedirect(QString name)
{
    return getValue(name, "DriveRedirect").toString();
}

QString ClientSettings::getDriveRedirectFolder(QString name)
{
    return getValue(name, "DriveRedirectFolder").toString();
}

QStringList ClientSettings::getDriveRedirectFolders(QString name)
{
    QString folders = getValue(name, "DriveRedirectFolders").toString();
    if ( folders.isEmpty() )
    {
        return QStringList();
    } else {
        return folders.split(" ");
    }
}

QStringList ClientSettings::getExcludedApps(QString name)
{
    QString excludedApps = getValue(name, "ExcludedApps").toString();
    qDebug() << excludedApps;
    if ( excludedApps.isEmpty() )
    {
        return QStringList();
    } else {
        return excludedApps.split(" ");
    }
}

QString ClientSettings::getUserWhiteList(QString name)
{
    return getValue(name, "UserWhiteList").toString();
}

QString ClientSettings::getUserBlackList(QString name)
{
    return getValue(name, "UserBlackList").toString();
}

bool ClientSettings::getSmartCardRedirect(QString name)
{
    return getValue(name, "SmartCardRedirect", QVariant(false)).toBool();
}

bool ClientSettings::getHomeRedirect(QString name)
{
    return getValue(name, "HomeRedirect", QVariant(false)).toBool();
}

bool ClientSettings::getRemovableRedirect(QString name)
{
    return getValue(name, "RemovableRedirect", QVariant(false)).toBool();
}

bool ClientSettings::getPrintersRedirect(QString name)
{
    return getValue(name, "PrintersRedirect", QVariant(false)).toBool();
}

bool ClientSettings::getSoundRedirect(QString name)
{
    return getValue(name, "SoundRedirect", QVariant(true)).toBool();
}

bool ClientSettings::getMicrophoneRedirect(QString name)
{
    return getValue(name, "MicrophoneRedirect", QVariant(true)).toBool();
}

bool ClientSettings::getWallpaper(QString name)
{
    return getValue(name, "EnableWallpaper", QVariant(false)).toBool();
}

bool ClientSettings::getEffects(QString name)
{
    return getValue(name, "EnableEffects", QVariant(false)).toBool();
}

QString ClientSettings::getRfxMode(QString name)
{
	return getValue(name, "RFXMode", "Video").toString();
}

bool ClientSettings::getMultimonSupport(QString name)
{
	return getValue(name, "EnableMultimon", QVariant(false)).toBool();
}

bool ClientSettings::getPortKnocker(QString name)
{
	return getValue(name, "PortKnocker", QVariant(false)).toBool();
}

QStringList ClientSettings::getExtraArgs(QString name)
{
	QStringList ret;

	QString extraArgs = getValue(name, "ExtraArgs", QVariant()).toString();

	if ( ! extraArgs.isEmpty() )
	{
		ret = extraArgs.split(" ");
		qDebug(ret.join(" ").toUtf8().data());
	} else {
		qDebug("ExtraArgs is empty! Setting default extra args");
		ret = QString("+rfx,+auto-reconnect,/gfx:AVC444:on,/cache:codec:rfx,/sound:sys:alsa").split(",");
	}

	return ret;
}

void ClientSettings::setLastLogin(QString lastLogin)
{
    userSettings->setValue("LastLogin", lastLogin);
}

void ClientSettings::setLastServer(QString lastServer)
{
    userSettings->setValue("LastServer", lastServer);
}

void ClientSettings::setLogin(QString name, QString login)
{
    userSettings->setValue(name + "/Login", login);
}

void ClientSettings::setServer(QString name, QString server)
{
    userSettings->setValue(name + "/Server", server);
}

void ClientSettings::setType(QString name, QString connType)
{
    userSettings->setValue(name + "/Type", connType);
}

void ClientSettings::setDomain(QString name, QString domain)
{
    userSettings->setValue(name + "/Domain", domain);
}

void ClientSettings::setPort(QString name, long port)
{
    userSettings->setValue(name + "/Port", (long long)port);
}

void ClientSettings::setScreenSize(QString name, QString screenSize)
{
    userSettings->setValue(name + "/ScreenSize", screenSize);
}

void ClientSettings::setOrientation(QString name, QString orientation)
{
    userSettings->setValue(name + "/Orientation", orientation);
}

void ClientSettings::setColorDepth(QString name, int colorDepth)
{
    userSettings->setValue(name + "/ColorDepth", colorDepth);
}

void ClientSettings::setDriveRedirect(QString name, QString redirectDrive)
{
    userSettings->setValue(name + "/DriveRedirect", redirectDrive);
}

void ClientSettings::setDriveRedirectFolder(QString name, QString folder)
{
    userSettings->setValue(name +"/DriveRedirectFolder", folder);
}

void ClientSettings::setDriveRedirectFolders(QString name, QStringList folders)
{
    if ( folders.count() > 0 )
    {
        userSettings->setValue(name +"/DriveRedirectFolders", folders.join(" "));
    } else {
        userSettings->setValue(name +"/DriveRedirectFolders", "");
    }
}

void ClientSettings::setExcludedApps(QString name, QStringList apps)
{
    if ( apps.count() > 0 )
    {
        userSettings->setValue(name +"/ExcludedApps", apps.join(" "));
    } else {
        userSettings->setValue(name +"/ExcludedApps", "");
    }
}

void ClientSettings::setUserWhiteList(QString name, QString userWhiteList)
{
    userSettings->setValue(name +"/UserWhiteList", userWhiteList);
}

void ClientSettings::setUserBlackList(QString name, QString userBlackList)
{
    userSettings->setValue(name +"/UserBlackList", userBlackList);
}

void ClientSettings::setSmartCardRedirect(QString name, bool redirectSmartCard)
{
    userSettings->setValue(name + "/SmartCardRedirect", redirectSmartCard);
}

void ClientSettings::setHomeRedirect(QString name, bool redirectHome)
{
    userSettings->setValue(name + "/HomeRedirect", redirectHome);
}

void ClientSettings::setRemovableRedirect(QString name, bool redirectRemovable)
{
    userSettings->setValue(name + "/RemovableRedirect", redirectRemovable);
}

void ClientSettings::setPrintersRedirect(QString name, bool redirectPrinters)
{
    userSettings->setValue(name + "/PrintersRedirect", redirectPrinters);
}

void ClientSettings::setSoundRedirect(QString name, bool redirectSound)
{
    userSettings->setValue(name + "/SoundRedirect", redirectSound);
}

void ClientSettings::setMicrophoneRedirect(QString name, bool redirectMicrophone)
{
    userSettings->setValue(name + "/MicrophoneRedirect", redirectMicrophone);
}

void ClientSettings::setWallpaper(QString name, bool enableWallpaper)
{
    userSettings->setValue(name + "/EnableWallpaper", enableWallpaper);
}

void ClientSettings::setEffects(QString name, bool enableEffects)
{
    userSettings->setValue(name + "/EnableEffects", enableEffects);
}

void ClientSettings::setRfxMode(QString name, QString rfxMode)
{
	userSettings->setValue(name + "/RFXMode", rfxMode);
}

void ClientSettings::setMultimonSupport(QString name, bool enableMultimon)
{
	userSettings->setValue(name + "/EnableMultimon", enableMultimon);
}

void ClientSettings::setExtraArgs(QString name, QStringList args)
{
	userSettings->setValue(name + "/ExtraArgs", args.join(" "));
}

QStringList ClientSettings::appNames()
{
    QStringList res;
    QStringList systemList = systemSettings->childGroups();
    for (int i = 0; i < systemList.count(); i++)
    {
        if (systemList[i].startsWith(APP_PREFIX, Qt::CaseInsensitive))
        {
            res += systemList[i].right(systemList[i].size() - APP_PREFIX.size() );
        }
    }
    return res;
}

QString ClientSettings::getAppExec(QString appName)
{
    return getValue(APP_PREFIX + appName, "Exec").toString();
}

QString ClientSettings::getAppExecArgs(QString appName)
{
    QString ret = getValue(APP_PREFIX + appName, "ExecArgs").toString();
    return ret;
}

QStringList ClientSettings::getAppExecEnv(QString appName)
{
    return getValue(APP_PREFIX + appName, "ExecEnv").toString().split(" ");
}

QString ClientSettings::getAppIcon(QString appName)
{
    return getValue(APP_PREFIX + appName, "Icon").toString();
}

QString ClientSettings::getAppWindowName(QString appName)
{
    return getValue(APP_PREFIX + appName, "WindowName").toString();
}

bool ClientSettings::isAppEnabled(QString appName)
{
    return ( getValue(APP_PREFIX + appName, "Enabled").toString().toLower() == "true" );
}
